------------------------------------------------------------------------------
--                                                                          --
--                            GNATPP COMPONENTS                             --
--                                                                          --
--                  G N A T P P . S O U R C E _ T A B L E                   --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                    Copyright (C) 2004-2009, AdaCore                      --
--                                                                          --
-- GNATPP is free software; you can redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNATPP is  distributed in the  hope that it will  be  useful, but --
-- WITHOUT ANY WARRANTY; without even the implied warranty of  MERCHANTABI- --
-- LITY or  FITNESS  FOR A  PARTICULAR  PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston,                                                                  --
--                                                                          --
-- GNATPP is maintained by ACT Europe (http://www.act-europe.fr).           --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the source file table - the table containing the
--  information about the source files to be processed and the state of their
--  processing

--  This file is very close to METRICS.Source_Table. We should think about
--  factoring out common components of ASIS tools library

package GNATPP.Source_Table is
   Low_SF_Bound   : constant := 0;
   High_SF_Bound : constant := 999_999;
   --  Almost 1_000_000 source files for one run of the tool

   type SF_Id is range Low_SF_Bound .. High_SF_Bound;

   No_SF_Id    : constant SF_Id := Low_SF_Bound;
   First_SF_Id : constant SF_Id := No_SF_Id + 1;

   Total_Sources : Natural;
   Sources_Left  : Natural;
   --  Counters used to form and output progress information.

   type SF_Status is (
      Waiting,
      --  Waiting for processing
      Not_A_Legal_Source,
      --  The file does not contain the compilable source

      Error_Detected,
      --  Some gnatpp problem has been created when reformatting this source
      --  so the results of reformatting are not safe

      Out_File_Problem,
      --  The out file was not successfully created or opened

      Processed);
      --  The source file has been successfully reformatted

   function Present (SF : SF_Id) return Boolean;
   --  Checks that SF is not is equal to No_SF_Id

   function File_Find (Full_SF_Name : String) return SF_Id;
   --  Returns the Id of the file with full name Full_SF_Name stored in the
   --  files table. Returns No_SF_Id if the table does not contain such file.

   procedure Set_New_SF_Record;
   --  Sets the initial values for the new source file record
   --  ??? Do we need this?

   procedure Add_Source_To_Process
     (Fname       : String;
      No_Argument : out Boolean);
   --  Fname is treated as the name of the file to process by gnatpp. If it is
   --  an empty string, this procedure set No_Argument ON (to stop iteration
   --  through tool parameters) and does else. Otherwise it tries to add this
   --  file to the table of files to process. The following checks are
   --  performed:
   --
   --  - first, this routine checks if Fname is the name of some existing file,
   --    and if it is not, generates the corresponding diagnosis and does
   --    nothing else;
   --
   --  - then, it checks if we already have stored a file with the same name.
   --    If we have the file with the same name, but from a different
   --    directory, the corresponding warning is generated, but the file is
   --    added to the file table (the situation when gnatpp is called
   --    to process files with the same name but from different directories
   --    looks strange, but this may be quite legal and reasonable). But if we
   --    have already stored in the list the name of exactly the same file, we
   --    generate the error message and do not change anything in the list of
   --    files.
   --
   --  At this stage we do not know if it is a compilable Ada source file.
   --
   --  If Fname is not an empty string, sets No_Argument_File_Specified OFF

   function Last_Source return SF_Id;
   --  Returns the Id of the last source stored in the source table. Returns
   --  No_SF_Id if there is no source file stored

   procedure Reset_Source_Iterator;
   function Next_Non_Processed_Source
     (Only_Bodies : Boolean := False)
      return        SF_Id;
   --  These two routines provide the iterator through the sources stored in
   --  the sources table. Reset_Source_Iterator resets the iterator to the
   --  ID of the first source stored in the table. Next_Non_Processed_Source
   --  returns the Id if the next source file stored in the file table which
   --  has not been processed yet. If Only_Bodies parameter is set ON, only
   --  body files are considered. We are not 100% sure that this file indeed
   --  is an Ada body file, at this stage we consider any file which name has
   --  ".adb" suffix as a body file. No_SF_Id is returned if there is no such
   --  file in the file table. The idea behind is to process first all the body
   --  files and for each body to process as much specs as possible using this
   --  body tree file, to minimize a set of trees needed to be created to
   --  process all the sources.

   procedure Init;
   --  Initializes the internal data structures

   ------------------------------------------------
   -- General source file access/update routines --
   ------------------------------------------------

   function Source_Name       (SF : SF_Id) return String;
   function Short_Source_Name (SF : SF_Id) return String;
   function Source_Status     (SF : SF_Id) return SF_Status;
   function Suffixless_Name   (SF : SF_Id) return String;
   --  Returns the file name with no directory information and with
   --  no suffix (if any).

   procedure Set_Source_Status (SF : SF_Id; S : SF_Status);

   ----------------------
   -- Problem counters --
   ----------------------

   Illegal_Sources   : Natural := 0;
   Tool_Failures     : Natural := 0;
   Out_File_Problems : Natural := 0;

end GNATPP.Source_Table;
