------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--                      A S I S _ U L . S T R I N G S                       --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                    Copyright (C) 2004-2009, AdaCore                      --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING.  If  not,  write  to  the Free Software Foundation, 51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package provides a string storage mechanism for varied length strings
--  Note that this mechanism is less efficient then direct using of access to
--  String values (for example GNAT.OS_Lib.String_Access). But from the other
--  side it allows to store and reuse varied length strings.

with Asis; use Asis;

package ASIS_UL.Strings is

   type String_Loc is record
      First, Last : Natural;
   end record;
   --  This record contains the start and end positions of a string inside
   --  a character table

   Nil_String_Loc : String_Loc := (0, 0);
   --  Corresponds to an empty string

   function Enter_String (S : String) return String_Loc;
   --  Stores a string in a character array, returning its starting and ending
   --  positions in a String_Loc structure

   function Get_String (SL : String_Loc) return String;
   --  Retrieves a string from a character array, based on its starting
   --  and ending positions supplied by SL

   function Is_Equal (S : String; SL : String_Loc) return Boolean;
   --  Checks if S is equal to a string represented by SL. Returns False if
   --  SL represents null string of if SL does not represent a string stored
   --  in the string storage.

   procedure Init;
   --  Resets the string table

   function Build_GNAT_Location
     (For_Elem : Asis.Element;
      Line     : Natural := 0;
      Column   : Natural := 0)
      return     String_Loc;
   --  Builds the string that describes the Element location in the form
   --  "file_name:line_number. For Elements from the instantiations
   --  the chain location_of_instantiation - location_in_template is created.
   --
   --  If For_Elem is Is_Part_Of_Implicit Elements and represents an implicit
   --  inherited subprogram or component thereof, the created SLOC has the
   --  format SLOC1(SLOC2), where SLOC1 is a SLOC of the corresponding
   --  explicit subprogram (or corresponding component thereof) from which
   --  this implicit subprogram has been inherited (directly or throufg a
   --  chain of other *implicit* derivations), and SLOC2 corresponds to the
   --  derived type declaration that is the "owner" of this implicit
   --  subprogram.
   --
   --  For any other Is_Part_Of_Implicit Element this function cannot produce
   --  any meaningful result.
   --
   --  Parameters Line and Column may have non-zero values only if the
   --  argument Element is not from expanded instantiation. If they are set,
   --  then the corresponding values are used as line and column numbers in
   --  the source location, otherwise line and column numbers are computed
   --  from the argument Element.
   --
   --  The the argument Element is from package Standard, the generated string
   --  is "Standard location", indepentently on the actuals for Line and
   --  Column parameters

   function Build_GNAT_Location
     (For_Elem : Asis.Element;
      Line     : Natural := 0;
      Column   : Natural := 0)
      return     String;
   --  Similar to the previous function, but this function returns the string
   --  result without allocating any information in the string storage.

end ASIS_UL.Strings;
