------------------------------------------------------------------------------
--                                                                          --
--                            GNATPP COMPONENTS                             --
--                                                                          --
--  A S I S _ U L . E N V I R O N M E N T . S C A N _ P A R A M E T E R S   --
--                                                                          --
--              (adapted for gnatpp from ASIS Utility Library)              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2009, AdaCore                        --
--                                                                          --
-- GNATPP is free software; you can redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNATPP is  distributed in the  hope that it will  be  useful, but --
-- WITHOUT ANY WARRANTY; without even the implied warranty of  MERCHANTABI- --
-- LITY or  FITNESS  FOR A  PARTICULAR  PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write to the Free Software Foundation,  51 Franklin Street, Fifth Floor, --
-- Boston,                                                                  --
--                                                                          --
-- GNATPP is maintained by AdaCore (http://www.adacore.com)                 --
--                                                                          --
------------------------------------------------------------------------------

with GNAT.Command_Line;        use GNAT.Command_Line;

with ASIS_UL.Compiler_Options;
with ASIS_UL.Misc;             use ASIS_UL.Misc;
with ASIS_UL.Options;          use ASIS_UL.Options;
with ASIS_UL.Source_Table;     use ASIS_UL.Source_Table;

with GNATPP.Dictionaries;
with GNATPP.Options;           use GNATPP.Options;
with GNATPP.Output;            use GNATPP.Output;
with GNATPP.State;             use GNATPP.State;

separate (ASIS_UL.Environment)
procedure Scan_Parameters is
   No_More_Args : Boolean := True;
   Nat          : Natural;
begin
   Warning_Mode := Quiet;
   --  Otherwise an average gnatpp run would generate too much diagnoses about
   --  problems in reformatting

   Process_RTL_Units := True;
   --  gnatpp does not care about this

   PP_Cont_Line_Indentation_Set := False;

   Initialize_Option_Scan
     (Stop_At_First_Non_Switch => True,
      Section_Delimiters       => "cargs");

   loop

      case
         GNAT.Command_Line.Getopt
           ("aL aU aM cl! c0 c1 c2 c3 c4 c5 "    &
            "e ff i! kL kU l1 l2 l3 M! "         &
            "hx "                                &
            "N notab nL nU nM nD D- D: "         &
            "pL pU pM "                          &
            "A0 A1 A2 A3 A4 A5 T! "              &
            "-no-separate-is "                   &
            "-separate-label "                   &
            "-separate-loop-then "               &
            "-no-separate-loop-then "            &
            "-use-on-new-line "                  &
            "-separate-stmt-name "               &
            "-RM-style-spacing "                 &
            "I: gnatec! -RTS= v w q dd "         &
            "gnat05 "                            & --  Ada 2005 mode
            --  output file control
            "-eol= files= pipe o= of= r rf rnb " &
            --  encoding of the result file(s)
            "Wh Wu Ws We W8 Wb")
      is
         when ASCII.NUL =>
            exit;

         when 'A' =>

            if Full_Switch = "A0" then
               Alignment_On := False;
            elsif Full_Switch = "A1" then
               Align_Colons_In_Decl := True;
            elsif Full_Switch = "A2" then
               Align_Colons_In_Decl := True;
               Align_Asign_In_Decl := True;
            elsif Full_Switch = "A3" then
               Align_Asign_In_Stmts := True;
            elsif Full_Switch = "A4" then
               Align_Arrows := True;
            elsif Full_Switch = "A5" then
               Allign_Ats := True;
            end if;

         when 'a' =>

            if Full_Switch = "aL" then
               PP_Attribute_Casing := Lower_Case;
            elsif Full_Switch = "aU" then
               PP_Attribute_Casing := Upper_Case;
            elsif Full_Switch = "aM" then
               PP_Attribute_Casing := Mixed;
            end if;

         when 'c' =>

            if Full_Switch = "c0" then
               Format_Comments := False;
            elsif Full_Switch = "c1" then
               GNAT_Comment_Inden      := True;
               Standard_Comment_Indent := False;
            elsif Full_Switch = "c2" then
               GNAT_Comment_Inden      := False;
               Standard_Comment_Indent := True;
            elsif Full_Switch = "c3" then
               GNAT_Comment_Start := True;
            elsif Full_Switch = "c4" then
               Reformat_Comment_Block := True;
            elsif Full_Switch = "c5" then
               Preserve_Special_Comments := True;
            elsif Full_Switch = "cl" then
               Nat := Get_Nat_Switch_Parameter (Parameter);

               if Nat not in 1 .. 9 then
                  Error ("wrong continuation line indentation (" &
                         Parameter & ")");
                  raise Parameter_Error;
               else
                  PP_Cont_Line_Indentation := Nat;
                  PP_Cont_Line_Indentation_Set := True;
               end if;

            end if;

         when 'D' =>

            if Full_Switch = "D-" then
               Use_Predefined_Casing := False;
            else
               Use_Dictionary := True;
               GNATPP.Dictionaries.Scan_Dictionary (Parameter);
            end if;

         when 'd' =>
            Progress_Indicator_Mode := True;

         when 'e' =>
            End_Labels := False;

         when 'f' =>

            if Full_Switch = "files" then
               Read_Args_From_File (Parameter, Store_With_No_Check => True);
            elsif Full_Switch = "ff" then
               Add_FF := True;
            end if;

         when 'h' =>

            if Full_Switch = "hx" then
               ASIS_UL.Options.Generate_XML_Help := True;
            end if;

         when 'i' =>
            Nat := Get_Nat_Switch_Parameter (Parameter);

            if Nat not in 1 .. 9 then
               Error ("wrong indentation (" & Parameter & ")");
               raise Parameter_Error;
            else
               PP_Indentation := Nat;
            end if;

         when 'k' =>

            if Full_Switch = "kL" then
               PP_Keyword_Casing := Lower_Case;
            elsif Full_Switch = "kU" then
               PP_Keyword_Casing := Upper_Case;
            end if;

         when 'l' =>

            if Full_Switch = "l1" then
               Compact_Layout  := True;
               Add_Empty_Lines := True;
            elsif Full_Switch = "l2" then
               Compact_Layout  := True;
               Add_Empty_Lines := False;
            elsif Full_Switch = "l3" then
               Compact_Layout  := False;
               Add_Empty_Lines := False;
            end if;

         when 'M' =>

            Nat := Get_Nat_Switch_Parameter (Parameter);

            if Nat not in 32 .. 256 then
               Put      ("gnatpp: wrong max line length (");
               Put      (Parameter);
               Put_Line (")");
               raise Parameter_Error;
            else
               Max_Line_Length := Nat;
            end if;

         when 'n' =>

            if Full_Switch = "nL" then
               PP_Name_Casing := Lower_Case;
            elsif Full_Switch = "nU" then
               PP_Name_Casing := Upper_Case;
            elsif Full_Switch = "nM" then
               PP_Name_Casing := Mixed;
            elsif Full_Switch = "nD" then
               PP_Name_Casing := As_Declared;
            elsif Full_Switch = "notab" then
               --  -notab is an obsolete feature, replaced with -N
               No_Tab_In_Comments := True;
            end if;

         when 'N' =>
            No_Tab_In_Comments := True;

         when 'p' =>

            if Full_Switch = "pL" then
               PP_Pragma_Casing := Lower_Case;
            elsif Full_Switch = "pU" then
               PP_Pragma_Casing := Upper_Case;
            elsif Full_Switch = "pM" then
               PP_Pragma_Casing := Mixed;
            elsif Full_Switch = "pipe" then
               Output_Mode := Pipe;
            end if;

         when 'q' =>
            Quiet_Mode := True;

         when 'r' =>

            if Full_Switch = "r" then
               Output_Mode := Replace;
            elsif Full_Switch = "rnb" then
               Output_Mode := Replace_No_Backup;
            elsif Full_Switch = "rf" then
               Output_Mode := Force_Replace;
            end if;

         when 'T' =>

            Case_Threshold := Get_Nat_Switch_Parameter (Parameter);

         when 'I' | 'g' | '-' =>

            if Full_Switch = "I" then
               Store_I_Option (Parameter);
            elsif Full_Switch = "gnatec" then
               Store_GNAT_Option_With_Path (Full_Switch, Parameter);
            elsif Full_Switch = "-RTS" then
               Store_Option ("--RTS=" & Parameter);

            elsif Full_Switch = "-eol" then
               Out_File_Format := Get_Out_File_Format (Parameter);
            elsif Full_Switch = "-no-separate-is" then
               Separate_Line_For_IS := False;
            elsif Full_Switch = "-separate-label" then
               Separate_Line_For_Label := True;
            elsif Full_Switch = "-separate-loop-then" then
               Separate_Line_For_THEN_and_LOOP := True;
            elsif Full_Switch = "-no-separate-loop-then" then
               No_Separate_Line_For_THEN_and_LOOP := True;
            elsif Full_Switch = "-use-on-new-line" then
               Separate_Line_For_USE := True;
            elsif Full_Switch = "-separate-stmt-name" then
               Separate_Line_For_Stmt_Name := True;
            elsif Full_Switch = "-RM-style-spacing" then
               RM_Style_Spacing := True;
            elsif Full_Switch = "gnat05" then
               ASIS_UL.Options.ASIS_2005_Mode := True;
            else
               Brief_Help;
               raise Parameter_Error;
            end if;

         when 'o' =>

            if Full_Switch = "o" then
               Output_Mode := Create_File;
            elsif Full_Switch = "of" then
               Output_Mode := Force_Create_File;
            else
               Brief_Help;
               raise Parameter_Error;
            end if;

            Res_File_Name := new String'(Parameter);

         when 'v' =>

            Verbose_Mode := True;
            Print_Version_Info (2003);

         when 'W' =>

            if Full_Switch = "Wh" then
               Output_Encoding := Hex_ESC;
            elsif Full_Switch = "Wu" then
               Output_Encoding := Upper_Half;
            elsif Full_Switch = "Ws" then
               Output_Encoding := Shift_JIS;
            elsif Full_Switch = "We" then
               Output_Encoding := EUC;
            elsif Full_Switch = "W8" then
               Output_Encoding := UTF_8;
            elsif Full_Switch = "Wb" then
               Output_Encoding := Brackets;
            end if;

         when 'w' =>

            Warning_Mode := Full;

         when others =>
            Brief_Help;
            raise Parameter_Error;
      end case;

   end loop;

   loop
      Store_Sources_To_Process
        (Get_Argument (Do_Expansion => True), No_More_Args);
      exit when No_More_Args;
   end loop;

   Process_cargs_Section (No_Preprocessing => True);

exception
   when GNAT.Command_Line.Invalid_Switch =>
      Error ("invalid switch : " & Full_Switch);
      Brief_Help;
      raise Parameter_Error;

   when GNAT.Command_Line.Invalid_Parameter =>
      Error ("parameter missed for : -" & Full_Switch);
      Brief_Help;
      raise Parameter_Error;

end Scan_Parameters;
