------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                  G N A T C H E C K . E X E M P T I O N                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2009, AdaCore                        --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

pragma Ada_2005;

with Ada.Characters.Handling;    use Ada.Characters.Handling;
--  with Ada.Wide_Text_IO;           use Ada.Wide_Text_IO;

with GNAT.OS_Lib;                use GNAT.OS_Lib;

with Asis.Elements;              use Asis.Elements;
with Asis.Expressions;           use Asis.Expressions;
with Asis.Text;                  use Asis.Text;

with ASIS_UL.Misc;               use ASIS_UL.Misc;
with ASIS_UL.Output;             use ASIS_UL.Output;

with Gnatcheck.Ids;              use Gnatcheck.Ids;
with Gnatcheck.Rules;            use Gnatcheck.Rules;
with Gnatcheck.Rules.Rule_Table; use Gnatcheck.Rules.Rule_Table;

--  with Gnatcheck.Ids;           use Gnatcheck.Ids;

package body Gnatcheck.Exemption is

   type Exemption_Kinds is
     (Not_An_Exemption,
      Exempt_On,
      Exempt_Off);

   function Get_Exemption_Kind (Image : Wide_String) return Exemption_Kinds;
   --  Returns Exemption_Kinds value represented by Image. Returns
   --  Not_An_Exemption if Image does not represent a valid exemption kind.

   ------------------------------------
   -- Check_Unclosed_Rule_Exemptions --
   ------------------------------------

   procedure Check_Unclosed_Rule_Exemptions (SF : SF_Id) is
   begin
      --  Very simple-minded implementation, diagnoses are not ordered
      --  according to SLOCs of unclosed exemptions

      for Rule in First_Rule .. All_Rules.Last loop

         if Is_Exempted (Rule) then
            SLOC_Error
              (Message => "No matching 'exempt_OFF' annotation for rule " &
                          Rule_Name (Rule),
               SLOC    => Short_Source_Name (SF)        & ':' &
                          Image (Exemption_Line (Rule)) & ':' &
                          Image (Exemption_Col  (Rule)));

            if Exemption_Violations (Rule) = 0 then
               SLOC_Error
                 (Message => "no detection for rule " & Rule_Name (Rule),
                  SLOC    => Short_Source_Name (SF)        & ':' &
                             Image (Exemption_Line (Rule)) & ':' &
                             Image (Exemption_Col  (Rule)));
            end if;

            Turn_Off_Exemption (Rule);
         end if;

      end loop;

   end Check_Unclosed_Rule_Exemptions;

   ---------------------
   -- Clean_Exemption --
   ---------------------

   procedure Clean_Exemption (Exemp : in out Exemption_Info) is
   begin
      Exemp.Line          := 0;
      Exemp.Col           := 0;
      Exemp.Justification := Nil_String_Loc;
      Exemp.Detected      := 0;
   end Clean_Exemption;

   -----------------------------
   -- Exemption_Justification --
   -----------------------------

   function Exemption_Justification
     (Exemp : Exemption_Info)
      return  String_Loc
   is
   begin
      return Exemp.Justification;
   end Exemption_Justification;

   -------------------
   -- Exemption_Col --
   -------------------

   function Exemption_Col (Exemp : Exemption_Info) return Natural is
   begin
      return Exemp.Col;
   end Exemption_Col;

   --------------------
   -- Exemption_Line --
   --------------------

   function Exemption_Line (Exemp : Exemption_Info) return Natural is
   begin
      return Exemp.Line;
   end Exemption_Line;

   --------------------------
   -- Exemption_Violations --
   --------------------------

   function Exemption_Violations (Exemp : Exemption_Info) return Natural is
   begin
      return Exemp.Detected;
   end Exemption_Violations;

   ------------------------
   -- Get_Exemption_Kind --
   ------------------------

   function Get_Exemption_Kind (Image : Wide_String) return Exemption_Kinds is
      Result : Exemption_Kinds;
   begin

      if Image (Image'First) = '"' then
         Result :=
           Exemption_Kinds'Wide_Value
             (Image (Image'First + 1 .. Image'Last - 1));
      --  Old format of Annotate pragma. We have to cut out quotation marks
      else
         Result :=
           Exemption_Kinds'Wide_Value (Image);
      end if;

      return Result;
   exception
      when Constraint_Error =>
         return Not_An_Exemption;
   end Get_Exemption_Kind;

   --------------------------------
   -- Increase_Violation_Counter --
   --------------------------------

   procedure Increase_Violation_Counter (Exemp : in out Exemption_Info) is
   begin
      Exemp.Detected := Exemp.Detected + 1;
   end Increase_Violation_Counter;

   -------------------------
   -- Is_Exemption_Pragma --
   -------------------------

   function Is_Exemption_Pragma (El : Asis.Element) return Boolean is
      Result : Boolean := False;
   begin

      if Pragma_Kind (El) = An_Implementation_Defined_Pragma
        and then
         To_Lower (To_String (Pragma_Name_Image (El))) = "annotate"
      then

         declare
            Pragma_Args : constant Asis.Element_List :=
              Pragma_Argument_Associations (El);
            --  Always non-empty for Annotate pragma!
            First_Par : Asis.Element;
         begin
            First_Par := Pragma_Args (Pragma_Args'First);
            First_Par := Actual_Parameter (First_Par);

            if To_Lower (To_String (Name_Image (First_Par))) = "gnatcheck" then
               Result := True;
            end if;
         end;

      end if;

      return Result;
   end Is_Exemption_Pragma;

   -------------
   -- Present --
   -------------

   function Present (Exemp : Exemption_Info) return Boolean is
   begin
      return Exemp.Line > 0;
   end Present;

   ------------------------------
   -- Process_Exemption_Pragma --
   ------------------------------

   procedure Process_Exemption_Pragma (El : Asis.Element) is
      Pragma_Args : constant Asis.Element_List :=
        Pragma_Argument_Associations (El);

      First_Idx : constant Natural := Pragma_Args'First;
      Next_Arg  :          Asis.Element;
      Tmp_Str   :          String_Access;
      Exem_Span :          Asis.Text.Span;

      Rule           : Rule_Id;
      Exemption_Kind : Exemption_Kinds;
   begin

      --  First, analyse the pragma format:
      --
      --  1. Check that we have at least three parameters

      if Pragma_Args'Length < 3 then
         SLOC_Error ("too few parameters for exemption, ignored", El);
         return;
      end if;

      --  2. Second parameter should be either "Exempt_On" or "Exempt_Off"

      Next_Arg := Pragma_Args (First_Idx + 1);
      Next_Arg := Actual_Parameter (Next_Arg);

      if Expression_Kind (Next_Arg) = A_String_Literal then
         Exemption_Kind := Get_Exemption_Kind (Value_Image (Next_Arg));
      elsif Expression_Kind (Next_Arg) = An_Identifier then
         Exemption_Kind := Get_Exemption_Kind (Name_Image (Next_Arg));
      end if;

      if Exemption_Kind = Not_An_Exemption then
         SLOC_Error ("wrong exemption kind, ignored", Next_Arg);
         return;
      end if;

      --  3. Third parameter should be the name of some existing rule:

      Next_Arg := Pragma_Args (First_Idx + 2);
      Next_Arg := Actual_Parameter (Next_Arg);

      if Expression_Kind (Next_Arg) = A_String_Literal then
         Tmp_Str := new String'(To_String (Value_Image (Next_Arg)));
         Rule    := Get_Rule (Tmp_Str (Tmp_Str'First + 1 .. Tmp_Str'Last - 1));
         --  We have to cut out quotation marks

         Free (Tmp_Str);
      else
         Rule := No_Rule;
      end if;

      if No (Rule) then
         SLOC_Error ("wrong rule name in exemption, ignored", Next_Arg);
         return;
      end if;

      --  4. Fourth parameter, if present, should be a string.

      if Pragma_Args'Length >= 4 then
         Next_Arg := Pragma_Args (First_Idx + 3);
         Next_Arg := Actual_Parameter (Next_Arg);

         if Expression_Kind (Next_Arg) = A_String_Literal then
            Tmp_Str := new String'(To_String (Value_Image (Next_Arg)));
         end if;

         if Tmp_Str = null then
            SLOC_Error
              ("exemption justification should be a string", Next_Arg);
         end if;

         --  5. Fourth parameter is ignored if exemption is turned OFF

         if Exemption_Kind = Exempt_Off then
            SLOC_Error
              ("turning exemption OFF does not need justification", Next_Arg);
         end if;

      end if;

      --  6. If exemption is turned ON, justification is expected

      if Exemption_Kind = Exempt_On
        and then
         Pragma_Args'Length = 3
      then
         SLOC_Error ("turning exemption ON expects justification", El);
      end if;

      if Pragma_Args'Length >= 5 then
         Next_Arg := Pragma_Args (First_Idx + 4);
         SLOC_Error
           ("rule exemption may have at most four parameters", Next_Arg);
      end if;

      if not Is_Enable (All_Rules.Table (Rule).all) then
         --  If Rule does not denote the enabled rule - nothing to do
         return;
      end if;

      --  Now - processing of the exemption pragma:

      case Exemption_Kind is
         when Exempt_On =>
            if Is_Exempted (Rule) then
               SLOC_Error
                 ("rule " & Rule_Name (Rule) & " is already exempted at line" &
                  Exemption_Line (Rule)'Img, El);
               return;
            end if;

            Exem_Span := Element_Span (El);

            if Tmp_Str = null then
               Tmp_Str := new String'("""unjustified""");
            end if;

            Set_Rule_Exemption_State
              (For_Rule      => Rule,
               Line          => Exem_Span.First_Line,
               Col           => Exem_Span.First_Column,
               Justification => Enter_String (Tmp_Str
                 (Tmp_Str'First + 1 .. Tmp_Str'Last - 1)));

            Free (Tmp_Str);

         when Exempt_Off =>

            if not Is_Exempted (Rule) then
               SLOC_Error
                 ("rule " & Rule_Name (Rule) & " is not in exempted state",
                  El);
               return;
            end if;

            if Exemption_Violations (Rule) = 0 then
               SLOC_Error
                 ("no detection for "                           &
                  Rule_Name (Rule)                              &
                  " rule in exemption section starting at line" &
                  Exemption_Line (Rule)'Img,
                  El);
            end if;

            Turn_Off_Exemption (Rule);

         when Not_An_Exemption =>
            pragma Assert (False);
            null;
      end case;

   end Process_Exemption_Pragma;

   ------------------------
   -- Set_Exemption_Info --
   ------------------------

   procedure Set_Exemption_Info
     (Line          :     Natural := 0;
      Col           :     Natural := 0;
      Justification :     String_Loc := Nil_String_Loc;
      Value         : out Exemption_Info)
   is
   begin
      Value.Line          := Line;
      Value.Col           := Col;
      Value.Justification := Justification;
   end Set_Exemption_Info;

end Gnatcheck.Exemption;
